// scrape-agents.js

require('dotenv').config();
const fs        = require('fs');
const path      = require('path');
const os        = require('os');
const mysql     = require('mysql2/promise');
const readline  = require('readline');
const puppeteer = require('puppeteer-core');

const delay = ms => new Promise(res => setTimeout(res, ms));

const SOURCE_NAME = 'Sahibinden';
const dbConfig = {
  host:     'localhost',
  user:     'root',
  password: '',
  database: 'emlak_db',
  charset:  'utf8mb4'
};

// Konsoldan ENTER bekleyen yardımcı
function waitForEnter(prompt) {
  return new Promise(resolve => {
    const rl = readline.createInterface({ input: process.stdin, output: process.stdout });
    rl.question(prompt, () => { rl.close(); resolve(); });
  });
}

;(async () => {
  // 1) MySQL’e bağlan
  const conn = await mysql.createConnection(dbConfig);
  console.log('✅ MySQL’e bağlanıldı.');

  // 2) detailLinks.json’dan ilk 5 ilan linkini al
  const linksPath = path.resolve(__dirname, 'detailLinks.json');
  if (!fs.existsSync(linksPath)) {
    console.error('Error: detailLinks.json bulunamadı.');
    process.exit(1);
  }
  const detailLinks = JSON.parse(fs.readFileSync(linksPath, 'utf8')).slice(0, 5);

  // 3) Gerçek Chrome’a bağlan
  console.log('▶ 127.0.0.1:9222 üzerinden gerçek Chrome’a bağlanılıyor…');
  const browser = await puppeteer.connect({ browserURL: 'http://127.0.0.1:9222' });
  const pages   = await browser.pages();
  const page    = pages[0] || await browser.newPage();
  await page.setViewport({ width: 1200, height: 900 });

  // 4) Sahibinden ana sayfa, CF geçiş kontrolü
  console.log('▶ Sahibinden ana sayfa yükleniyor…');
  await page.goto('https://www.sahibinden.com', { waitUntil: 'networkidle2' });
  await delay(2000);

  if (page.url().includes('secure.sahibinden.com')) {
    console.log('\n⚠️ Cloudflare doğrulaması gerekiyor.');
    console.log('   Tarayıcıda tamamlayıp ENTER’a basın.');
    await waitForEnter('👉 ENTER basıldı, devam ediliyor...\n');
  }

  // 5) Döngü: Her bir ilan için acente & ekip bilgisi
  for (const detailUrl of detailLinks) {
    console.log('▶ Detay:', detailUrl);
    await page.goto(detailUrl, { waitUntil: 'networkidle2' });
    await delay(1500);

    // ——— Acente linki bul ———
    let agentHref = await page.$eval(
      'div.user-info-store-name > a',
      a => a.href
    ).catch(() => null);
    if (!agentHref) {
      agentHref = await page.$eval(
        '#classifiedDetail div[class*=classifiedContactSeller] a',
        a => a.href
      ).catch(() => null);
    }
    if (!agentHref) {
      console.log('  ⚠️ Acente linki bulunamadı.');
      continue;
    }
    console.log('  → Acente URL:', agentHref);

    // ——— Acenta sayfasına geç: isim ve konum ———
    await page.goto(agentHref, { waitUntil: 'networkidle2' });
    await delay(1000);
    const storeDiv = await page.$('div.contact-info div.store-name');
    const agencyName = storeDiv
      ? await storeDiv.evaluate(el => el.textContent.trim())
      : '';
    const location = storeDiv
      ? await storeDiv.evaluate(el => el.getAttribute('data-main-location'))
      : '';
    console.log(`  → ${agencyName} — ${location}`);

    // ——— agents tablosuna kaydet ———
    const [agentRes] = await conn.execute(
      `INSERT INTO agents (source, agency_name, agent_url, location)
       VALUES (?, ?, ?, ?)`,
      [SOURCE_NAME, agencyName, agentHref, location]
    );
    const agentId = agentRes.insertId;

    // ——— Ekibimiz sayfası ———
    const { origin } = new URL(agentHref);
    const teamUrl = `${origin}/ekibimiz`;
    console.log('  ▶ Ekibimiz:', teamUrl);
    await page.goto(teamUrl, { waitUntil: 'networkidle2' });
    await delay(2000);

    // ——— Ekip üyelerini çek ve team_members tablosuna ekle ———
    const members = await page.$$eval(
      'div.edr-trigger.consultant',
      cards => cards.map(c => ({
        name:  c.querySelector('p.name')?.innerText.trim()  || '',
        tag:   c.querySelector('p.title')?.innerText.trim() || '',
        phone: c.querySelector('a.phone-trigger')?.getAttribute('data-phone') || ''
      }))
    ).catch(() => []);

    for (const m of members) {
      await conn.execute(
        `INSERT INTO team_members (agent_id, name, tag, phone)
         VALUES (?, ?, ?, ?)`,
        [agentId, m.name, m.tag, m.phone]
      );
    }
    console.log(`  ✅ ${members.length} ekip üyesi kaydedildi.`);
  }

  // 6) Temizlik ve kapanış
  await browser.disconnect();
  await conn.end();
  console.log('🎉 İşlem tamamlandı. MySQL’e kaydedildi.');
})();
